//
//  GeometryGamesModel.swift
//
//  Created by Jeff on 5/10/19.
//  Copyright © 2020 Jeff Weeks. All rights reserved.
//

import Foundation


//	To lock or not to lock?
//
//	On macOS, a CVDisplayLink runs its animation on a separate thread.
//	The main thread and the CVDisplayLink thread would both need
//	to access the ModelData, so for sure we'd need to use a lock
//	to prevent conflicts.
//
//	On iOS, a CADisplayLink (or an MTKView if we were using one)
//	runs its animation on the main thread, so in principle
//	we might not need to use a lock when accessing the ModelData.
//	Nevertheless, whenever the Swift code needs an unsafe pointer
//	to the ModelData, it seems like a good idea to force it
//	to check that pointer out explicitly, and then check it
//	back in again when it's done with it.  That way we ensure
//	that one set of modifications to the underlying ModelData
//	is complete before the next set of modifications begins.
//	We could manage the check-out / check-in either
//
//		1.	with a Bool
//		2.	with a lock
//
//	Either way, we'd ensure that check-outs and check-ins
//	occur in strict alternation.  (In case of a violation,
//	we'd get a deadlock when using a lock, or a manually
//	generated exit when using a Bool.)
//
//	Even though a Bool would be fine for all present purposes,
//	I nevertheless decided to use a lock to keep this code
//	as future proof as possible, in case we someday want to access
//	the same GeometryGamesModel from two different threads
//	(for example, in the "thinking thread" in Torus Chess).


class GeometryGamesModel {

	//	To use the ModelData, clients should check out a pointer,
	//	use it, and then check it back in again as soon as possible:
	//
	//		do {
	//			let md = theModel.lockModelData()
	//			...
	//			use md as desired
	//			...
	//			theModel.unlockModelData()
	//		}
	//
	//	The enclosing do{} block ensures that the reference
	//	to md is no longer visible after the call to unlockModelData().
	//
	private let itsModelData: UnsafeMutablePointer<ModelData>
	private let itsModelDataLock : NSLock
	
	init() {

		//	The app's C bridging header gives our Swift code
		//	access to the definition of the ModelData as a C struct.
		itsModelData = UnsafeMutablePointer<ModelData>.allocate(capacity: 1)
		SetUpModelData(itsModelData)

		itsModelDataLock = NSLock()
	}
	
	deinit {
	
		//	If we're getting dealloc'd, that should mean that
		//	no objects hold a pointer to us, and no objects
		//	hold itsModelDataLock.
		
		ShutDownModelData(itsModelData)
		itsModelData.deallocate()
	}

	func lockModelData() -> UnsafeMutablePointer<ModelData> {

		itsModelDataLock.lock()

		return itsModelData
	}
	
	func unlockModelData() {

		itsModelDataLock.unlock()
	}
}
